"use client"

import { useState, useEffect } from "react"
import { WithdrawalRequestsTable } from "@/components/withdrawal-requests-table"
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card"
import { getWithdrawalRequests, approveAgentWithdrawal, approveSuperAgentWithdrawal } from "@/services/api"

export default function WithdrawalRequestsPage() {
  const [withdrawalRequests, setWithdrawalRequests] = useState([])
  const [loading, setLoading] = useState(true)
  const [error, setError] = useState(null)

  const fetchRequests = async () => {
    setLoading(true)
    setError(null)
    try {
      const data = await getWithdrawalRequests()
      setWithdrawalRequests(data)
    } catch (err) {
      setError("Failed to fetch withdrawal requests.")
      console.error(err)
    } finally {
      setLoading(false)
    }
  }

  useEffect(() => {
    fetchRequests()
  }, [])

  const handleApprove = async (withdrawalId, type) => {
    console.log(`Approving ${type} withdrawal request ${withdrawalId}`)
    try {
      // Find the request object to get necessary details for the payload
      const requestToApprove = withdrawalRequests.find(req => req.withdrawal_id === withdrawalId);

      if (!requestToApprove) {
        throw new Error("Withdrawal request not found.");
      }

      // The prompt specified `const { customer_id } = req.body;` for approval endpoints.
      // Assuming `customer_id` here refers to the `requester_id` for the approval process.
      const payload = { customer_id: requestToApprove.requester_id };

      if (type === "agent") {
        await approveAgentWithdrawal(withdrawalId, payload)
      } else if (type === "super_agent") { // Use 'super_agent' as per sample response
        await approveSuperAgentWithdrawal(withdrawalId, payload)
      }
      alert(`Withdrawal request ${withdrawalId} (${type}) approved!`)
      fetchRequests() // Re-fetch requests to update the list
    } catch (err) {
      console.error("Approval failed:", err)
      alert("Failed to approve withdrawal request.")
    }
  }

  return (
    <div className="flex-1 space-y-4 p-8 pt-6">
      <div className="flex items-center justify-between space-y-2">
        <h2 className="text-3xl font-bold tracking-tight">Withdrawal Requests</h2>
      </div>
      <Card>
        <CardHeader>
          <CardTitle>Manage Commission Withdrawal Requests</CardTitle>
        </CardHeader>
        <CardContent>
          {loading && <p>Loading withdrawal requests...</p>}
          {error && <p className="text-red-500">{error}</p>}
          {!loading && !error && (
            <WithdrawalRequestsTable
              requests={withdrawalRequests}
              onApprove={handleApprove}
            />
          )}
        </CardContent>
      </Card>
    </div>
  )
}
