"use client"

import { useState, useEffect } from "react";
import https from "@/services/https";
import * as XLSX from "xlsx";

import { Badge } from "@/components/ui/badge"
import { Button } from "@/components/ui/button"
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card"
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table"
import { Search, Filter, Users, CreditCard, MapPin, Phone, Mail, TrendingUp, AlertTriangle, Download } from "lucide-react"
import { Input } from "@/components/ui/input"
import { AddCustomerModal } from "./modals/add-customer-modal"
import Link from "next/link"

export default function CustomersPage() {
  const [customersData, setCustomersData] = useState([]);

  const fetchCustomers = async () => {
    try {
      const token = localStorage.getItem("token");
      const response = await fetch(`${https.baseUrl}/customers`, {
        headers: {
          "x-auth-token": token,
        },
      });
      if (!response.ok) {
        throw new Error("Failed to fetch customers");
      }
      const data = await response.json();
      setCustomersData(data);
    } catch (error) {
      console.error("Error fetching customers:", error);
    }
  };

  const handleDownload = () => {
    const worksheet = XLSX.utils.json_to_sheet(customersData);
    const workbook = XLSX.utils.book_new();
    XLSX.utils.book_append_sheet(workbook, worksheet, "Customers");
    XLSX.writeFile(workbook, "customers-data.xlsx");
  };

  useEffect(() => {
    fetchCustomers();
  }, []);

  const getCreditScoreColor = (score) => {
    if (score >= 80) return "text-green-600"
    if (score >= 60) return "text-yellow-600"
    return "text-red-600"
  }

  const getCreditScoreLabel = (score) => {
    if (score >= 80) return "Excellent"
    if (score >= 60) return "Good"
    return "Poor"
  }

  return (
    <div className="space-y-6">
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-3xl font-bold">Customers</h1>
          <p className="text-muted-foreground">Manage customer profiles and loan portfolios</p>
        </div>
        <AddCustomerModal onCustomerAdded={fetchCustomers} />
      </div>

      {/* Summary Cards */}
      <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-4">
        <Card className="shadow-md rounded-lg border hover:shadow-lg transition-shadow">
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Total Customers</CardTitle>
            <Users className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold text-blue-600">{customersData.length}</div>
            <p className="text-xs text-muted-foreground">{/* Dynamic data */}</p>
          </CardContent>
        </Card>
        <Card className="shadow-md rounded-lg border hover:shadow-lg transition-shadow">
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Active Customers</CardTitle>
            <TrendingUp className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold text-green-600">{customersData.filter(c => Number(c?.activeLoans) > 0).length}</div>
            <p className="text-xs text-muted-foreground">{/* Dynamic data */}</p>
          </CardContent>
        </Card>
        <Card className="shadow-md rounded-lg border hover:shadow-lg transition-shadow">
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Overdue Customers</CardTitle>
            <AlertTriangle className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold text-red-600">{customersData.filter(c => c.status === 'Overdue').length}</div>
            <p className="text-xs text-muted-foreground">{/* Dynamic data */}</p>
          </CardContent>
        </Card>
        <Card className="shadow-md rounded-lg border hover:shadow-lg transition-shadow">
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Avg Credit Score</CardTitle>
            <CreditCard className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold">{(customersData.reduce((acc, c) => acc + (c.creditScore || 0), 0) / customersData.length).toFixed(0)}</div>
            <p className="text-xs text-muted-foreground">{/* Dynamic data */}</p>
          </CardContent>
        </Card>
      </div>

      {/* Customer Table */}
      <Card>
        <CardHeader>
          <CardTitle>Customer Directory</CardTitle>
          <CardDescription>View and manage all customer profiles and loan information</CardDescription>
          <div className="flex items-center space-x-2">
            <div className="relative flex-1">
              <Search className="absolute left-2 top-2.5 h-4 w-4 text-muted-foreground" />
              <Input placeholder="Search customers..." className="pl-8" />
            </div>
            <Button className="bg-brand-primary text-white hover:bg-brand-primary/80" size="sm">
              <Filter className="mr-2 h-4 w-4" />
              Filter
            </Button>
            <Button variant="outline" size="sm" onClick={handleDownload}>
              <Download className="mr-2 h-4 w-4" />
              Download
            </Button>
          </div>
        </CardHeader>
        <CardContent>
          <Table>
            <TableHeader className="bg-gray-100">
              <TableRow>
                <TableHead>Customer</TableHead>
                <TableHead>Contact</TableHead>
                <TableHead>Location</TableHead>
                <TableHead>Credit Score</TableHead>
                <TableHead>Loans</TableHead>
                <TableHead>Outstanding</TableHead>
                {/* <TableHead>Status</TableHead> */}
                <TableHead>Actions</TableHead>
              </TableRow>
            </TableHeader>
            <TableBody className="text-blue-900">
              {customersData.length > 0 && customersData.map((customer) => (
                <TableRow key={customer.id} className="hover:bg-gray-100 even:bg-gray-50">
                  <TableCell>
                    <div>
                      <div className="font-medium text-blue-900">{customer.name}</div>
                      <div className="text-sm text-blue-700">{customer.id}</div>
                      <div className="text-xs text-blue-500">Joined: {new Date(customer.joinDate).toLocaleDateString()}</div>
                    </div>
                  </TableCell>
                  <TableCell>
                    <div className="space-y-1">
                      <div className="flex items-center text-sm text-blue-900">
                        <Phone className="mr-1 h-3 w-3" />
                        {customer.phone}
                      </div>
                      <div className="flex items-center text-sm text-blue-700">
                        <Mail className="mr-1 h-3 w-3" />
                        {customer.email}
                      </div>
                    </div>
                  </TableCell>
                  <TableCell>
                    <div className="flex items-center">
                      <MapPin className="mr-1 h-3 w-3" />
                      <div>
                        <div className="text-sm text-blue-900">{customer.location}</div>
                        <div className="text-xs text-blue-700">{customer.county}</div>
                        <div className="text-xs text-blue-700">{customer.gps}</div>

                      </div>
                    </div>
                  </TableCell>
                  <TableCell>
                    <div className="space-y-1">
                      <div className={`text-lg font-bold ${getCreditScoreColor(customer.creditScore)}`}>
                        {customer.creditScore}
                      </div>
                      <div className="text-xs text-muted-foreground">{getCreditScoreLabel(customer.creditScore)}</div>
                    </div>
                  </TableCell>
                  <TableCell>
                    <div className="space-y-1">
                      <div className="text-sm font-medium text-blue-900">
                        {customer.activeLoans} active / {customer.totalLoans} total
                      </div>
                      <div className="text-xs text-blue-700">{customer.devices} devices</div>
                    </div>
                  </TableCell>
                  <TableCell>
                    <div className="space-y-1">
                      <div className="font-medium text-blue-900">NGN {customer.outstandingBalance ? parseFloat(customer.outstandingBalance).toLocaleString() : 0}</div>
                      {customer.nextPaymentDue && (
                        <div className="text-xs text-blue-700">Next: {new Date(customer.nextPaymentDue).toLocaleDateString()}</div>
                      )}
                    </div>
                  </TableCell>
                  {/* <TableCell>
                    <Badge
                      variant={
                        customer.status === "Active"
                          ? "success"
                          : customer.status === "Overdue"
                            ? "destructive"
                            : customer.status === "New"
                              ? "secondary"
                              : "outline"
                      }
                    >
                      {customer.status}
                    </Badge>
                  </TableCell> */}
                  <TableCell>
                    <Link href={`/customers/${customer.id}`}>
                      <Button variant="outline" size="sm">
                        View Profile
                      </Button>
                    </Link>
                  </TableCell>
                </TableRow>
              ))}
            </TableBody>
          </Table>
        </CardContent>
      </Card>
    </div>
  )
}

