"use client";

import { useState, useEffect } from "react";
import { Button } from "@/components/ui/button";
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogFooter,
  DialogHeader,
  DialogTitle,
} from "@/components/ui/dialog";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import https from "@/services/https";
import Swal from "sweetalert2";

export function ChangeUsernameModal({ agent, isOpen, onOpenChange, onUsernameChanged }) {
  const [username, setUsername] = useState("");
  const [isLoading, setIsLoading] = useState(false);

  useEffect(() => {
    if (agent) {
      setUsername(agent.username || "");
    }
  }, [agent]);

  const handleClose = () => {
    onOpenChange(false);
  }

  const handleUsernameChange = (e) => {
    const value = e.target.value.toLowerCase().replace(/\s/g, '');
    setUsername(value);
  }

  const handleSubmit = async () => {
    if (!username) {
      Swal.fire("Error", "Username cannot be empty", "error");
      return;
    }

    setIsLoading(true);
    try {
      const token = localStorage.getItem("token");
      const response = await fetch(`${https.baseUrl}/admin/users/${agent.id}/change-username`, {
        method: "PUT",
        headers: {
          "Content-Type": "application/json",
          "x-auth-token": token,
        },
        body: JSON.stringify({ username }),
      });

      if (!response.ok) {
        const errorData = await response.json();
        throw new Error(errorData.msg || "Failed to change username");
      }

      Swal.fire("Success", "Username changed successfully", "success");
      if(onUsernameChanged) {
        onUsernameChanged();
      }
      handleClose();
    } catch (error) {
      console.error("Error changing username:", error);
      Swal.fire("Error", error.message, "error");
    } finally {
      setIsLoading(false);
    }
  };

  return (
    <Dialog open={isOpen} onOpenChange={onOpenChange}>
      <DialogContent className="sm:max-w-[425px]" onInteractOutside={handleClose}>
        <DialogHeader>
          <DialogTitle>Change Username for {agent?.name}</DialogTitle>
          <DialogDescription>
            Enter a new username for the agent. No spaces allowed.
          </DialogDescription>
        </DialogHeader>
        <div className="grid gap-4 py-4">
          <div className="grid grid-cols-4 items-center gap-4">
            <Label htmlFor="new-username" className="text-right">
              New Username
            </Label>
            <Input
              id="new-username"
              type="text"
              value={username}
              onChange={handleUsernameChange}
              className="col-span-3"
            />
          </div>
        </div>
        <DialogFooter>
          <Button variant="outline" onClick={handleClose}>Cancel</Button>
          <Button type="submit" onClick={handleSubmit} disabled={isLoading}>
            {isLoading ? "Saving..." : "Save Changes"}
          </Button>
        </DialogFooter>
      </DialogContent>
    </Dialog>
  );
}
