'use client'

import { useState, useEffect } from "react"
import { Badge } from "@/components/ui/badge"
import { Button } from "@/components/ui/button"
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card"
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs"
import { Dialog, DialogContent } from "@/components/ui/dialog"
import { Input } from "@/components/ui/input"
import { Label } from "@/components/ui/label"
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select"
import {
  ArrowLeft,
  DollarSign,
  User,
  Phone,
  Mail,
  MapPin,
  Calendar,
  Package,
  Landmark,
  Wallet,
  History,
  Banknote,
  Pencil,
} from "lucide-react"
import Link from "next/link"
import { AddCreditModal } from "./modals/add-credit-modal";
import https from "@/services/https";
import CreditSummary from "./credit-summary";
import { getBankAccount, createBankAccount, getBanks } from "@/services/api";

export default function AgentDetailPage({ agentId }) {
  const [agent, setAgent] = useState(null)
  const [isLoading, setIsLoading] = useState(false)
  const [error, setError] = useState(null)
  const [isAvatarModalOpen, setIsAvatarModalOpen] = useState(false);
  const [isGeneratingAccount, setIsGeneratingAccount] = useState(false);
  const [bankAccount, setBankAccount] = useState(null);
  const [banks, setBanks] = useState([]);
  const [newAccountDetails, setNewAccountDetails] = useState({ bank_code: '', account_number: '' });
  const [isBankAccountLoading, setIsBankAccountLoading] = useState(false);
  const [isBanksLoading, setIsBanksLoading] = useState(false);
  const [bankAccountError, setBankAccountError] = useState(null);
  const [isEditingBankAccount, setIsEditingBankAccount] = useState(false);

  const fetchAgentData = async () => {
    setIsLoading(true)
    try {
      const token = localStorage.getItem("token")
      const response = await fetch(`${https.baseUrl}/agents/${agentId}`, {
        headers: {
          "x-auth-token": token,
        },
      })
      if (!response.ok) {
        throw new Error("Failed to fetch agent data")
      }
      const data = await response.json()
      setAgent(data)
    } catch (err) {
      setError(err.message)
      console.error("Error fetching agent:", err)
    } finally {
      setIsLoading(false)
    }
  }

  const fetchBankAccount = async () => {
    setIsBankAccountLoading(true);
    setBankAccountError(null);
    try {
      const data = await getBankAccount(agentId);
      setBankAccount(data);
      if (data) {
        setNewAccountDetails({ bank_code: data.bank_code, account_number: data.account_number });
      }
    } catch (err) {
      setBankAccountError("Failed to fetch bank account.");
      console.error(err);
    } finally {
      setIsBankAccountLoading(false);
    }
  };

  const fetchBanks = async () => {
    setIsBanksLoading(true);
    try {
      const data = await getBanks();
      setBanks(data);
    } catch (err) {
      setBankAccountError("Failed to fetch banks.");
      console.error(err);
    } finally {
      setIsBanksLoading(false);
    }
  };

  const handleCreateBankAccount = async (e) => {
    e.preventDefault();
    setIsBankAccountLoading(true);
    setBankAccountError(null);
    try {
      const data = await createBankAccount(agentId, newAccountDetails);
      setBankAccount(data);
      setIsEditingBankAccount(false);
    } catch (err) {
      setBankAccountError("Failed to create/update bank account.");
      console.error(err);
    } finally {
      setIsBankAccountLoading(false);
    }
  };

  const generateAccountNumber = async () => {
    setIsGeneratingAccount(true);
    try {
      const token = localStorage.getItem("token");
      const response = await fetch(`${https.baseUrl}/agents/${agentId}/dedicated-account`, {
        method: 'POST',
        headers: {
          "x-auth-token": token,
        },
      });
      if (!response.ok) {
        throw new Error("Failed to generate account number");
      }
      // Refresh agent data to get the new account number
      fetchAgentData();
    } catch (err) {
      setError(err.message);
      console.error("Error generating account number:", err);
    } finally {
      setIsGeneratingAccount(false);
    }
  };

  useEffect(() => {
    if (agentId) {
      fetchAgentData();
      fetchBankAccount();
      fetchBanks();
    }
  }, [agentId])

  if (isLoading || !agent) {
    return <div>Loading agent data...</div>
  }

  if (error) {
    return <div>Error: {error}</div>
  }

  return (
    <div className="space-y-6">
      {/* Header */}
      <div className="flex items-center justify-between">
        <div className="flex items-center space-x-4">
          <Link href="/agents">
            <Button className="bg-brand-primary text-white hover:bg-brand-primary/80" size="sm">
              <ArrowLeft className="mr-2 h-4 w-4" />
              Back to Agents
            </Button>
          </Link>
          <div className="flex items-center space-x-4">
            <div className="w-16 h-16 rounded-full cursor-pointer" onClick={() => setIsAvatarModalOpen(true)}>
                {agent.profile_img ? (
                    <img
                        src={`data:image/jpeg;base64,${agent.profile_img}`}
                        alt={agent.name}
                        className="w-16 h-16 rounded-full object-cover"
                    />
                ) : (
                    <div className="w-16 h-16 bg-gray-200 rounded-full flex items-center justify-center">
                        <User className="h-8 w-8 text-gray-500" />
                    </div>
                )}
            </div>
            <div>
                <h1 className="text-3xl font-bold">{agent.name}</h1>
                <p className="text-muted-foreground">Agent ID: {agent.id}</p>
            </div>
          </div>
        </div>
        <div className="flex items-center space-x-2">
          <AddCreditModal userId={agent.id} userType="agent" onCreditAdded={fetchAgentData} />
          <Badge
            variant={
              agent.status === "active" ? "default" : "secondary"
            }
          >
            {agent.status}
          </Badge>
          {/* <EditAgentModal agent={agent} onUpdate={fetchAgentData} /> */}
        </div>
      </div>

      {/* Agent Stats Cards */}
      <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-4">
        <Card className="shadow-md rounded-lg border hover:shadow-lg transition-shadow">
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Total Sales</CardTitle>
            <DollarSign className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold text-green-600">NGN {agent.totalSales?.toLocaleString() || "0.00"}</div>
            <p className="text-xs text-muted-foreground">Total value of loans managed</p>
          </CardContent>
        </Card>

        <Card className="shadow-md rounded-lg border hover:shadow-lg transition-shadow">
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Devices Managed</CardTitle>
            <Package className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold text-blue-600">{agent.devicesManaged}</div>
            <p className="text-xs text-muted-foreground">Devices assigned by this agent</p>
          </CardContent>
        </Card>

        <Card className="shadow-md rounded-lg border hover:shadow-lg transition-shadow">
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Commission Rate</CardTitle>
            <Landmark className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold text-purple-600">{agent.commissionRate}%</div>
            <p className="text-xs text-muted-foreground">Current commission rate</p>
          </CardContent>
        </Card>

        <Card className="shadow-md rounded-lg border hover:shadow-lg transition-shadow">
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Commission Balance</CardTitle>
            <Wallet className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold text-green-600">NGN {agent.commissionBalance?.toLocaleString() || "0.00"}</div>
            <p className="text-xs text-muted-foreground">Available for withdrawal</p>
          </CardContent>
        </Card>

        <Card className="shadow-md rounded-lg border hover:shadow-lg transition-shadow">
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Credit Balance</CardTitle>
            <Wallet className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            <div className={`text-2xl font-bold ${parseFloat(agent.credit_balance) > 0 ? "text-green-600" : "text-red-600"}`}>NGN {agent.credit_balance?.toLocaleString() || "0.00"}</div>
            <p className="text-xs text-muted-foreground">Agent's wallet balance</p>
          </CardContent>
        </Card>
      </div>

      {/* Main Content */}
      <div className="grid gap-6 md:grid-cols-3">
        <div className="md:col-span-2">
          <Tabs defaultValue="overview" className="space-y-4">
            <TabsList>
              <TabsTrigger value="overview" className="data-[state=active]:bg-brand-primary data-[state=active]:text-white data-[state=inactive]:bg-transparent data-[state=inactive]:text-gray-500 data-[state=inactive]:hover:bg-gray-200 transition-colors">Overview</TabsTrigger>
              <TabsTrigger value="devices" className="data-[state=active]:bg-brand-primary data-[state=active]:text-white data-[state=inactive]:bg-transparent data-[state=inactive]:text-gray-500 data-[state=inactive]:hover:bg-gray-200 transition-colors">Assigned Devices</TabsTrigger>
              <TabsTrigger value="withdrawals" className="data-[state=active]:bg-brand-primary data-[state=active]:text-white data-[state=inactive]:bg-transparent data-[state=inactive]:text-gray-500 data-[state=inactive]:hover:bg-gray-200 transition-colors">Withdrawal History</TabsTrigger>
              <TabsTrigger value="credit-summary" className="data-[state=active]:bg-brand-primary data-[state=active]:text-white data-[state=inactive]:bg-transparent data-[state=inactive]:text-gray-500 data-[state=inactive]:hover:bg-gray-200 transition-colors">Credit Summary</TabsTrigger>
              <TabsTrigger value="bank-account" className="data-[state=active]:bg-brand-primary data-[state=active]:text-white data-[state=inactive]:bg-transparent data-[state=inactive]:text-gray-500 data-[state=inactive]:hover:bg-gray-200 transition-colors">Bank Account</TabsTrigger>
            </TabsList>

            <TabsContent value="overview" className="space-y-4">
              <Card className="shadow-md rounded-lg border hover:shadow-lg transition-shadow">
                <CardHeader>
                  <CardTitle>Agent Information</CardTitle>
                </CardHeader>
                <CardContent className="space-y-3">
                    <div className="flex items-center space-x-2">
                      <Phone className="h-4 w-4 text-blue-900" />
                      <span className="text-sm text-blue-700">{agent.phone}</span>
                    </div>
                    <div className="flex items-center space-x-2">
                      <Mail className="h-4 w-4 text-blue-900" />
                      <span className="text-sm text-blue-700">{agent.email}</span>
                    </div>
                    <div className="flex items-center space-x-2">
                      <MapPin className="h-4 w-4 text-blue-900" />
                      <span className="text-sm text-blue-700">{agent.address}, {agent.city}, {agent.region}</span>
                    </div>
                    <div className="flex items-center space-x-2">
                      <Landmark className="h-4 w-4 text-blue-900" />
                      <span className="text-sm text-blue-700">Landmark: {agent.landmark}</span>
                    </div>
                    <div className="flex items-center space-x-2">
                      <Calendar className="h-4 w-4 text-blue-900" />
                      <span className="text-sm text-blue-700">Joined: {new Date(agent.joinDate).toLocaleDateString()}</span>
                    </div>
                    <div className="flex items-center space-x-2">
                      <History className="h-4 w-4 text-blue-900" />
                      <span className="text-sm text-blue-700">Last Active: { agent.lastActive ? new Date(agent.lastActive).toLocaleString() : 'N/A'}</span>
                    </div>
                    <div className="flex items-center space-x-2">
                      <Wallet className="h-4 w-4 text-blue-900" />
                      {agent.accountNumber ? (
                        <span className="text-sm">Account No: {agent.accountNumber}</span>
                      ) : (
                        <Button onClick={generateAccountNumber} disabled={isGeneratingAccount} className="bg-green-500 text-white hover:bg-green-600">
                          {isGeneratingAccount ? 'Generating...' : 'Generate Account Number'}
                        </Button>
                      )}
                    </div>
                </CardContent>
              </Card>
            </TabsContent>

            <TabsContent value="devices" className="space-y-4">
              <Card className="shadow-md rounded-lg border hover:shadow-lg transition-shadow">
                <CardHeader>
                  <CardTitle>Assigned Devices</CardTitle>
                </CardHeader>
                <CardContent>
                  {agent.assignedDevices && agent.assignedDevices.length > 0 ? (
                    <div className="grid gap-4 md:grid-cols-2">
                      {agent.assignedDevices.map((device) => (
                        <Card key={device.id}>
                          <CardHeader>
                            <CardTitle className="text-lg text-blue-900">{device.serialNumber}</CardTitle>
                            <Badge variant={device.status === "assigned" ? "success" : "secondary"}>
                              {device.status}
                            </Badge>
                          </CardHeader>
                          <CardContent className="space-y-2">
                            <p className="text-sm text-blue-700">Customer: {device.customerName}</p>
                            <p className="text-sm text-blue-500">Install Date: {new Date(device.installDate).toLocaleDateString()}</p>
                          </CardContent>
                        </Card>
                      ))}
                    </div>
                  ) : (
                    <p className="text-muted-foreground">No devices assigned by this agent.</p>
                  )}
                </CardContent>
              </Card>
            </TabsContent>

            <TabsContent value="withdrawals" className="space-y-4">
              <Card className="shadow-md rounded-lg border hover:shadow-lg transition-shadow">
                <CardHeader>
                  <CardTitle>Withdrawal History</CardTitle>
                </CardHeader>
                <CardContent>
                  {agent.withdrawalHistory && agent.withdrawalHistory.length > 0 ? (
                    <div className="space-y-4">
                      {agent.withdrawalHistory.map((withdrawal) => (
                        <div key={withdrawal.id} className="flex items-center justify-between border-b pb-2">
                          <div>
                            <p className="font-medium text-blue-900">NGN {withdrawal.amount.toLocaleString()}</p>
                            <p className="text-sm text-blue-700">{new Date(withdrawal.date).toLocaleDateString()}</p>
                          </div>
                          <Badge variant="outline">{withdrawal.transactionId || "N/A"}</Badge>
                        </div>
                      ))}
                    </div>
                  ) : (
                    <p className="text-muted-foreground">No withdrawal history available.</p>
                  )}
                </CardContent>
              </Card>
            </TabsContent>

            <TabsContent value="credit-summary">
              <CreditSummary userId={agentId} userType="agent" />
            </TabsContent>

            <TabsContent value="bank-account" className="space-y-4">
              <Card className="shadow-md rounded-lg border hover:shadow-lg transition-shadow">
                <CardHeader className="flex flex-row items-center justify-between">
                  <CardTitle>Commission Bank Account</CardTitle>
                  {bankAccount && !isEditingBankAccount && (
                    <Button variant="outline" size="sm" onClick={() => setIsEditingBankAccount(true)}>
                      <Pencil className="h-4 w-4 mr-2" />
                      Edit
                    </Button>
                  )}
                </CardHeader>
                <CardContent>
                  {isBankAccountLoading || isBanksLoading ? (
                    <p>Loading bank details...</p>
                  ) : bankAccountError ? (
                    <p className="text-red-500">{bankAccountError}</p>
                  ) : (isEditingBankAccount || !bankAccount) ? (
                    <form onSubmit={handleCreateBankAccount} className="space-y-4">
                      <p>
                        {isEditingBankAccount ? "Update the agent's bank account." : "No bank account set up for this agent."}
                      </p>
                      <div className="grid gap-2">
                        <Label htmlFor="bank">Bank</Label>
                        <Select
                          value={newAccountDetails.bank_code}
                          onValueChange={(value) => setNewAccountDetails({ ...newAccountDetails, bank_code: value })}
                          required
                        >
                          <SelectTrigger>
                            <SelectValue placeholder="Select a bank" />
                          </SelectTrigger>
                          <SelectContent>
                            {banks.map((bank) => (
                              <SelectItem key={bank.code} value={bank.code}>
                                {bank.name}
                              </SelectItem>
                            ))}
                          </SelectContent>
                        </Select>
                      </div>
                      <div className="grid gap-2">
                        <Label htmlFor="account_number">Account Number</Label>
                        <Input
                          id="account_number"
                          value={newAccountDetails.account_number}
                          onChange={(e) => setNewAccountDetails({ ...newAccountDetails, account_number: e.target.value })}
                          placeholder="Enter account number"
                          required
                        />
                      </div>
                      <div className="flex space-x-2">
                        <Button type="submit" disabled={isBankAccountLoading}>
                          {isBankAccountLoading ? 'Saving...' : 'Save Bank Account'}
                        </Button>
                        {isEditingBankAccount && (
                          <Button variant="ghost" onClick={() => setIsEditingBankAccount(false)}>
                            Cancel
                          </Button>
                        )}
                      </div>
                    </form>
                  ) : (
                    <div className="space-y-2">
                      <div className="flex items-center space-x-2">
                        <Banknote className="h-4 w-4 text-blue-900" />
                        <span className="font-semibold">Bank:</span>
                        <span>{bankAccount.bank_name}</span>
                      </div>
                      <div className="flex items-center space-x-2">
                        <User className="h-4 w-4 text-blue-900" />
                        <span className="font-semibold">Account Name:</span>
                        <span>{bankAccount.account_name}</span>
                      </div>
                      <div className="flex items-center space-x-2">
                        <Wallet className="h-4 w-4 text-blue-900" />
                        <span className="font-semibold">Account Number:</span>
                        <span>{bankAccount.account_number}</span>
                      </div>
                    </div>
                  )}
                </CardContent>
              </Card>
            </TabsContent>

          </Tabs>
        </div>

        {/* Quick Actions Sidebar */}
        {/* <AgentQuickActions agent={agent} onAction={handleAgentAction} isLoading={isLoading} /> */}
      </div>
      <Dialog open={isAvatarModalOpen} onOpenChange={setIsAvatarModalOpen}>
        <DialogContent className="max-w-md">
            {agent.profile_img ? (
                <img
                    src={`data:image/jpeg;base64,${agent.profile_img}`}
                    alt={agent.name}
                    className="w-full h-auto rounded-lg"
                />
            ) : (
                <div className="w-full h-auto bg-gray-200 rounded-lg flex items-center justify-center" style={{ aspectRatio: '1 / 1' }}>
                    <User className="h-32 w-32 text-gray-500" />
                </div>
            )}
        </DialogContent>
      </Dialog>
    </div>
  )
}
